//
//  IntegrationTests.swift
//  death_app Watch AppTests
//
//  Created by Task Master on 2025-09-16.
//

import XCTest
import HealthKit
@testable import death_app_Watch_App

final class IntegrationTests: XCTestCase {
    
    var predictionEngine: PredictionEngine!
    var healthKitService: HealthKitService!
    var actuarialModel: ActuarialModel!
    var mockHealthStore: MockHKHealthStore!
    
    override func setUpWithError() throws {
        super.setUp()
        
        mockHealthStore = MockHKHealthStore()
        healthKitService = HealthKitService(healthStore: mockHealthStore)
        actuarialModel = ActuarialModel()
        predictionEngine = PredictionEngine(
            healthKitService: healthKitService,
            actuarialModel: actuarialModel
        )
    }
    
    override func tearDownWithError() throws {
        predictionEngine = nil
        healthKitService = nil
        actuarialModel = nil
        mockHealthStore = nil
        super.tearDown()
    }
    
    // MARK: - Full Prediction Pipeline Tests
    
    func testCompleteHealthDataToPredictionFlow() throws {
        let expectation = XCTestExpectation(description: "Complete prediction flow")
        
        // Setup realistic mock health data
        setupRealisticHealthData()
        
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -45, to: Date())!,
            gender: .male,
            height: 175.0,
            weight: 75.0
        )
        
        predictionEngine.generatePrediction(for: userProfile) { result in
            switch result {
            case .success(let prediction):
                XCTAssertNotNil(prediction.lifeExpectancy, "Should generate life expectancy")
                XCTAssertGreaterThan(prediction.lifeExpectancy, userProfile.currentAge, "Life expectancy should be greater than current age")
                XCTAssertLessThan(prediction.lifeExpectancy, 120.0, "Life expectancy should be realistic")
                XCTAssertNotNil(prediction.riskFactors, "Should include risk factors")
                XCTAssertNotNil(prediction.healthMetrics, "Should include health metrics")
                XCTAssertNotNil(prediction.generatedAt, "Should have generation timestamp")
                
            case .failure(let error):
                XCTFail("Prediction should succeed: \(error)")
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 10.0)
    }
    
    func testPredictionWithMinimalHealthData() throws {
        let expectation = XCTestExpectation(description: "Prediction with minimal data")
        
        // Setup minimal health data
        setupMinimalHealthData()
        
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -30, to: Date())!,
            gender: .female
        )
        
        predictionEngine.generatePrediction(for: userProfile) { result in
            switch result {
            case .success(let prediction):
                XCTAssertNotNil(prediction.lifeExpectancy)
                XCTAssertTrue(prediction.riskFactors.isEmpty || prediction.riskFactors.count > 0, "Should handle minimal data")
                
            case .failure:
                XCTFail("Should handle minimal data gracefully")
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 5.0)
    }
    
    func testPredictionWithRichHealthData() throws {
        let expectation = XCTestExpectation(description: "Prediction with rich data")
        
        // Setup comprehensive health data
        setupComprehensiveHealthData()
        
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -35, to: Date())!,
            gender: .male,
            height: 180.0,
            weight: 80.0
        )
        
        predictionEngine.generatePrediction(for: userProfile) { result in
            switch result {
            case .success(let prediction):
                XCTAssertNotNil(prediction.lifeExpectancy)
                XCTAssertGreaterThan(prediction.riskFactors.count, 3, "Should include multiple risk factors")
                XCTAssertGreaterThan(prediction.healthMetrics.count, 5, "Should include multiple health metrics")
                XCTAssertNotNil(prediction.confidenceScore, "Should provide confidence score")
                XCTAssertGreaterThan(prediction.confidenceScore ?? 0, 0.5, "Should have reasonable confidence")
                
            case .failure(let error):
                XCTFail("Rich data prediction should succeed: \(error)")
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 10.0)
    }
    
    // MARK: - Data Flow Validation Tests
    
    func testHealthKitToActuarialModelDataFlow() throws {
        let expectation = XCTestExpectation(description: "HealthKit to Actuarial data flow")
        
        setupRealisticHealthData()
        
        // Test that HealthKit data properly flows to actuarial model
        healthKitService.fetchAllHealthMetrics(from: Date().addingTimeInterval(-2592000), to: Date()) { metrics, error in
            XCTAssertNil(error, "HealthKit fetch should succeed")
            XCTAssertNotNil(metrics, "Should receive health metrics")
            
            if let metrics = metrics {
                let riskFactors = self.predictionEngine.convertHealthMetricsToRiskFactors(metrics)
                XCTAssertFalse(riskFactors.isEmpty, "Should convert metrics to risk factors")
                
                let prediction = self.actuarialModel.calculateLifeExpectancy(age: 40.0, riskFactors: riskFactors)
                XCTAssertNotNil(prediction, "Should calculate life expectancy from health data")
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 5.0)
    }
    
    func testPredictionHistoryIntegration() throws {
        let expectation = XCTestExpectation(description: "Prediction history integration")
        
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -40, to: Date())!,
            gender: .female
        )
        
        // Generate multiple predictions to test history
        var completedPredictions = 0
        let totalPredictions = 3
        
        for i in 0..<totalPredictions {
            DispatchQueue.main.asyncAfter(deadline: .now() + Double(i) * 0.5) {
                self.predictionEngine.generatePrediction(for: userProfile) { result in
                    switch result {
                    case .success(let prediction):
                        // Verify prediction is stored in history
                        let history = PredictionHistory.shared
                        history.savePrediction(prediction)
                        
                        completedPredictions += 1
                        if completedPredictions == totalPredictions {
                            let storedPredictions = history.getAllPredictions()
                            XCTAssertGreaterThanOrEqual(storedPredictions.count, totalPredictions, "Should store all predictions")
                            
                            // Test trend analysis
                            let trends = history.analyzeTrends()
                            XCTAssertNotNil(trends, "Should provide trend analysis")
                            
                            expectation.fulfill()
                        }
                        
                    case .failure:
                        XCTFail("Prediction should succeed for history test")
                    }
                }
            }
        }
        
        wait(for: [expectation], timeout: 15.0)
    }
    
    // MARK: - Error Handling Integration Tests
    
    func testNetworkErrorRecovery() throws {
        let expectation = XCTestExpectation(description: "Network error recovery")
        
        mockHealthStore.shouldFailWithNetworkError = true
        
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -25, to: Date())!,
            gender: .male
        )
        
        predictionEngine.generatePrediction(for: userProfile) { result in
            switch result {
            case .success:
                XCTFail("Should fail with network error initially")
                
            case .failure(let error):
                XCTAssertNotNil(error, "Should receive network error")
                
                // Test recovery - disable network error and retry
                self.mockHealthStore.shouldFailWithNetworkError = false
                
                self.predictionEngine.generatePrediction(for: userProfile) { recoveryResult in
                    switch recoveryResult {
                    case .success:
                        XCTAssertTrue(true, "Should recover from network error")
                        
                    case .failure:
                        XCTFail("Should recover after network is restored")
                    }
                    
                    expectation.fulfill()
                }
            }
        }
        
        wait(for: [expectation], timeout: 10.0)
    }
    
    func testPermissionDeniedHandling() throws {
        let expectation = XCTestExpectation(description: "Permission denied handling")
        
        mockHealthStore.shouldFailWithPermissionError = true
        mockHealthStore.mockAuthorizationStatus = .sharingDenied
        
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -50, to: Date())!,
            gender: .female
        )
        
        predictionEngine.generatePrediction(for: userProfile) { result in
            switch result {
            case .success(let prediction):
                // Should still generate prediction with limited data
                XCTAssertNotNil(prediction.lifeExpectancy, "Should generate prediction without HealthKit")
                XCTAssertTrue(prediction.riskFactors.isEmpty || prediction.riskFactors.count < 3, "Should have limited risk factors")
                
            case .failure:
                // Acceptable if completely dependent on HealthKit
                XCTAssertTrue(true, "Can fail if HealthKit is required")
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 5.0)
    }
    
    // MARK: - Performance Integration Tests
    
    func testEndToEndPerformance() throws {
        setupComprehensiveHealthData()
        
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -45, to: Date())!,
            gender: .male,
            height: 175.0,
            weight: 78.0
        )
        
        measure {
            let expectation = XCTestExpectation(description: "End-to-end performance")
            
            predictionEngine.generatePrediction(for: userProfile) { _ in
                expectation.fulfill()
            }
            
            wait(for: [expectation], timeout: 5.0)
        }
    }
    
    func testConcurrentPredictions() throws {
        let expectation = XCTestExpectation(description: "Concurrent predictions")
        expectation.expectedFulfillmentCount = 5
        
        setupRealisticHealthData()
        
        let profiles = [
            UserProfile(birthDate: Calendar.current.date(byAdding: .year, value: -25, to: Date())!, gender: .male),
            UserProfile(birthDate: Calendar.current.date(byAdding: .year, value: -35, to: Date())!, gender: .female),
            UserProfile(birthDate: Calendar.current.date(byAdding: .year, value: -45, to: Date())!, gender: .male),
            UserProfile(birthDate: Calendar.current.date(byAdding: .year, value: -55, to: Date())!, gender: .female),
            UserProfile(birthDate: Calendar.current.date(byAdding: .year, value: -65, to: Date())!, gender: .male)
        ]
        
        for profile in profiles {
            DispatchQueue.global().async {
                self.predictionEngine.generatePrediction(for: profile) { result in
                    switch result {
                    case .success(let prediction):
                        XCTAssertNotNil(prediction.lifeExpectancy)
                        
                    case .failure:
                        XCTFail("Concurrent prediction should succeed")
                    }
                    
                    expectation.fulfill()
                }
            }
        }
        
        wait(for: [expectation], timeout: 15.0)
    }
    
    // MARK: - Helper Methods
    
    private func setupRealisticHealthData() {
        let heartRateType = HKQuantityType.quantityType(forIdentifier: .heartRate)!
        let stepCountType = HKQuantityType.quantityType(forIdentifier: .stepCount)!
        let activeEnergyType = HKQuantityType.quantityType(forIdentifier: .activeEnergyBurned)!
        
        var samples: [HKSample] = []
        
        // Add realistic heart rate data (60-80 bpm)
        for i in 0..<100 {
            let heartRate = 65.0 + Double.random(in: -10...15)
            let quantity = HKQuantity(unit: HKUnit(from: "count/min"), doubleValue: heartRate)
            let sample = HKQuantitySample(
                type: heartRateType,
                quantity: quantity,
                start: Date().addingTimeInterval(-Double(i * 300)),
                end: Date().addingTimeInterval(-Double(i * 300) + 60)
            )
            samples.append(sample)
        }
        
        // Add step count data (8000-12000 steps per day)
        for i in 0..<30 {
            let steps = 8000.0 + Double.random(in: 0...4000)
            let quantity = HKQuantity(unit: .count(), doubleValue: steps)
            let sample = HKQuantitySample(
                type: stepCountType,
                quantity: quantity,
                start: Calendar.current.startOfDay(for: Date().addingTimeInterval(-Double(i * 86400))),
                end: Date().addingTimeInterval(-Double(i * 86400))
            )
            samples.append(sample)
        }
        
        // Add active energy data
        for i in 0..<30 {
            let energy = 400.0 + Double.random(in: 0...200)
            let quantity = HKQuantity(unit: .kilocalorie(), doubleValue: energy)
            let sample = HKQuantitySample(
                type: activeEnergyType,
                quantity: quantity,
                start: Calendar.current.startOfDay(for: Date().addingTimeInterval(-Double(i * 86400))),
                end: Date().addingTimeInterval(-Double(i * 86400))
            )
            samples.append(sample)
        }
        
        mockHealthStore.mockSamples = samples
        mockHealthStore.shouldSucceedAuthorization = true
        mockHealthStore.mockAuthorizationStatus = .sharingAuthorized
    }
    
    private func setupMinimalHealthData() {
        // Only basic step count
        let stepCountType = HKQuantityType.quantityType(forIdentifier: .stepCount)!
        let quantity = HKQuantity(unit: .count(), doubleValue: 5000.0)
        let sample = HKQuantitySample(
            type: stepCountType,
            quantity: quantity,
            start: Calendar.current.startOfDay(for: Date()),
            end: Date()
        )
        
        mockHealthStore.mockSamples = [sample]
        mockHealthStore.shouldSucceedAuthorization = true
        mockHealthStore.mockAuthorizationStatus = .sharingAuthorized
    }
    
    private func setupComprehensiveHealthData() {
        setupRealisticHealthData()
        
        // Add additional comprehensive metrics
        let bodyMassType = HKQuantityType.quantityType(forIdentifier: .bodyMass)!
        let heightType = HKQuantityType.quantityType(forIdentifier: .height)!
        let restingHeartRateType = HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!
        
        var additionalSamples: [HKSample] = []
        
        // Body mass
        let massQuantity = HKQuantity(unit: .gramUnit(with: .kilo), doubleValue: 75.0)
        let massSample = HKQuantitySample(
            type: bodyMassType,
            quantity: massQuantity,
            start: Date().addingTimeInterval(-86400),
            end: Date().addingTimeInterval(-86400)
        )
        additionalSamples.append(massSample)
        
        // Height
        let heightQuantity = HKQuantity(unit: .meterUnit(with: .centi), doubleValue: 175.0)
        let heightSample = HKQuantitySample(
            type: heightType,
            quantity: heightQuantity,
            start: Date().addingTimeInterval(-86400),
            end: Date().addingTimeInterval(-86400)
        )
        additionalSamples.append(heightSample)
        
        // Resting heart rate
        let restingHRQuantity = HKQuantity(unit: HKUnit(from: "count/min"), doubleValue: 58.0)
        let restingHRSample = HKQuantitySample(
            type: restingHeartRateType,
            quantity: restingHRQuantity,
            start: Date().addingTimeInterval(-86400),
            end: Date().addingTimeInterval(-86400)
        )
        additionalSamples.append(restingHRSample)
        
        mockHealthStore.mockSamples.append(contentsOf: additionalSamples)
    }
}